<?php
session_start();
require_once 'config/database.php';
require_once 'config/session.php';

// Check if user is admin
if (!isAdmin()) {
    header('Location: dashboard.php');
    exit();
}

$current_page = 'settings';
$success_msg = '';
$error_msg = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['update_general'])) {
        // Update general settings
        $company_name = $_POST['company_name'];
        $company_address = $_POST['company_address'];
        $company_email = $_POST['company_email'];
        $company_phone = $_POST['company_phone'];
        
        // Store in database (you may need to create a settings table)
        try {
            $stmt = $pdo->prepare("
                INSERT INTO settings (setting_key, setting_value) 
                VALUES 
                ('company_name', ?),
                ('company_address', ?),
                ('company_email', ?),
                ('company_phone', ?)
                ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)
            ");
            $stmt->execute([$company_name, $company_address, $company_email, $company_phone]);
            
            $success_msg = "General settings updated successfully!";
        } catch (Exception $e) {
            $error_msg = "Error updating settings: " . $e->getMessage();
        }
    }
    
    if (isset($_POST['update_attendance'])) {
        // Update attendance settings
        $work_start_time = $_POST['work_start_time'];
        $work_end_time = $_POST['work_end_time'];
        $late_threshold = $_POST['late_threshold'];
        $early_leave_threshold = $_POST['early_leave_threshold'];
        
        try {
            $stmt = $pdo->prepare("
                INSERT INTO settings (setting_key, setting_value) 
                VALUES 
                ('work_start_time', ?),
                ('work_end_time', ?),
                ('late_threshold', ?),
                ('early_leave_threshold', ?)
                ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)
            ");
            $stmt->execute([$work_start_time, $work_end_time, $late_threshold, $early_leave_threshold]);
            
            $success_msg = "Attendance settings updated successfully!";
        } catch (Exception $e) {
            $error_msg = "Error updating settings: " . $e->getMessage();
        }
    }
    
    if (isset($_POST['update_leave'])) {
        // Update leave settings
        $leave_year_start = $_POST['leave_year_start'];
        $default_casual_leaves = $_POST['default_casual_leaves'];
        $default_sick_leaves = $_POST['default_sick_leaves'];
        $default_annual_leaves = $_POST['default_annual_leaves'];
        
        try {
            $stmt = $pdo->prepare("
                INSERT INTO settings (setting_key, setting_value) 
                VALUES 
                ('leave_year_start', ?),
                ('default_casual_leaves', ?),
                ('default_sick_leaves', ?),
                ('default_annual_leaves', ?)
                ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)
            ");
            $stmt->execute([$leave_year_start, $default_casual_leaves, $default_sick_leaves, $default_annual_leaves]);
            
            $success_msg = "Leave settings updated successfully!";
        } catch (Exception $e) {
            $error_msg = "Error updating settings: " . $e->getMessage();
        }
    }
    
    if (isset($_POST['update_email'])) {
        // Update email settings
        $smtp_host = $_POST['smtp_host'];
        $smtp_port = $_POST['smtp_port'];
        $smtp_username = $_POST['smtp_username'];
        $smtp_password = $_POST['smtp_password'];
        $from_email = $_POST['from_email'];
        $from_name = $_POST['from_name'];
        
        try {
            $stmt = $pdo->prepare("
                INSERT INTO settings (setting_key, setting_value) 
                VALUES 
                ('smtp_host', ?),
                ('smtp_port', ?),
                ('smtp_username', ?),
                ('smtp_password', ?),
                ('from_email', ?),
                ('from_name', ?)
                ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)
            ");
            $stmt->execute([$smtp_host, $smtp_port, $smtp_username, $smtp_password, $from_email, $from_name]);
            
            $success_msg = "Email settings updated successfully!";
        } catch (Exception $e) {
            $error_msg = "Error updating settings: " . $e->getMessage();
        }
    }
}

// Get current settings
$settings = [];
try {
    $stmt = $pdo->query("SELECT setting_key, setting_value FROM settings");
    $result = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    $settings = $result;
} catch (Exception $e) {
    // Settings table might not exist
    $settings = [];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Settings</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.1/font/bootstrap-icons.css">
    <style>
        .settings-card {
            margin-bottom: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .settings-card .card-header {
            background-color: #f8f9fa;
            border-bottom: 1px solid #e9ecef;
        }
        .form-label {
            font-weight: 500;
        }
        .settings-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 10px;
        }
    </style>
</head>
<body>
    <div class="container-fluid mt-4">
        <!-- Messages -->
        <?php if ($success_msg): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <i class="bi bi-check-circle"></i> <?php echo $success_msg; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>
        
        <?php if ($error_msg): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bi bi-exclamation-triangle"></i> <?php echo $error_msg; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>
        
        <div class="row">
            <div class="col-md-12">
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="bi bi-gear me-2"></i>System Settings</h5>
                    </div>
                    <div class="card-body">
                        <!-- Settings Tabs -->
                        <ul class="nav nav-tabs mb-4" id="settingsTab" role="tablist">
                            <li class="nav-item" role="presentation">
                                <button class="nav-link active" id="general-tab" data-bs-toggle="tab" data-bs-target="#general" type="button" role="tab">
                                    <i class="bi bi-building"></i> General
                                </button>
                            </li>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link" id="attendance-tab" data-bs-toggle="tab" data-bs-target="#attendance" type="button" role="tab">
                                    <i class="bi bi-clock"></i> Attendance
                                </button>
                            </li>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link" id="leave-tab" data-bs-toggle="tab" data-bs-target="#leave" type="button" role="tab">
                                    <i class="bi bi-calendar-event"></i> Leave
                                </button>
                            </li>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link" id="email-tab" data-bs-toggle="tab" data-bs-target="#email" type="button" role="tab">
                                    <i class="bi bi-envelope"></i> Email
                                </button>
                            </li>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link" id="system-tab" data-bs-toggle="tab" data-bs-target="#system" type="button" role="tab">
                                    <i class="bi bi-cpu"></i> System
                                </button>
                            </li>
                        </ul>
                        
                        <!-- Tab Content -->
                        <div class="tab-content" id="settingsTabContent">
                            
                            <!-- General Settings -->
                            <div class="tab-pane fade show active" id="general" role="tabpanel">
                                <div class="card settings-card">
                                    <div class="card-header">
                                        <h6 class="mb-0">Company Information</h6>
                                    </div>
                                    <div class="card-body">
                                        <form method="POST">
                                            <div class="row">
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Company Name</label>
                                                    <input type="text" name="company_name" class="form-control" 
                                                           value="<?php echo $settings['company_name'] ?? 'HR Management System'; ?>">
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Company Email</label>
                                                    <input type="email" name="company_email" class="form-control" 
                                                           value="<?php echo $settings['company_email'] ?? 'info@company.com'; ?>">
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Company Phone</label>
                                                    <input type="text" name="company_phone" class="form-control" 
                                                           value="<?php echo $settings['company_phone'] ?? '+94 11 234 5678'; ?>">
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Address</label>
                                                    <textarea name="company_address" class="form-control" rows="2"><?php echo $settings['company_address'] ?? 'Colombo, Sri Lanka'; ?></textarea>
                                                </div>
                                                <div class="col-12">
                                                    <button type="submit" name="update_general" class="btn btn-primary">
                                                        <i class="bi bi-save"></i> Save Changes
                                                    </button>
                                                </div>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Attendance Settings -->
                            <div class="tab-pane fade" id="attendance" role="tabpanel">
                                <div class="card settings-card">
                                    <div class="card-header">
                                        <h6 class="mb-0">Attendance Settings</h6>
                                    </div>
                                    <div class="card-body">
                                        <form method="POST">
                                            <div class="row">
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Work Start Time</label>
                                                    <input type="time" name="work_start_time" class="form-control" 
                                                           value="<?php echo $settings['work_start_time'] ?? '08:30'; ?>">
                                                    <small class="text-muted">Default work start time</small>
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Work End Time</label>
                                                    <input type="time" name="work_end_time" class="form-control" 
                                                           value="<?php echo $settings['work_end_time'] ?? '17:00'; ?>">
                                                    <small class="text-muted">Default work end time</small>
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Late Threshold (minutes)</label>
                                                    <input type="number" name="late_threshold" class="form-control" 
                                                           value="<?php echo $settings['late_threshold'] ?? 15; ?>" min="0" max="120">
                                                    <small class="text-muted">Minutes allowed before marking as late</small>
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Early Leave Threshold (minutes)</label>
                                                    <input type="number" name="early_leave_threshold" class="form-control" 
                                                           value="<?php echo $settings['early_leave_threshold'] ?? 15; ?>" min="0" max="120">
                                                    <small class="text-muted">Minutes allowed before marking as early leave</small>
                                                </div>
                                                <div class="col-12">
                                                    <button type="submit" name="update_attendance" class="btn btn-primary">
                                                        <i class="bi bi-save"></i> Save Changes
                                                    </button>
                                                </div>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Leave Settings -->
                            <div class="tab-pane fade" id="leave" role="tabpanel">
                                <div class="card settings-card">
                                    <div class="card-header">
                                        <h6 class="mb-0">Leave Settings</h6>
                                    </div>
                                    <div class="card-body">
                                        <form method="POST">
                                            <div class="row">
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Leave Year Start Month</label>
                                                    <select name="leave_year_start" class="form-control">
                                                        <?php for ($i = 1; $i <= 12; $i++): ?>
                                                        <option value="<?php echo $i; ?>" 
                                                            <?php echo ($settings['leave_year_start'] ?? 1) == $i ? 'selected' : ''; ?>>
                                                            <?php echo date('F', mktime(0, 0, 0, $i, 1)); ?>
                                                        </option>
                                                        <?php endfor; ?>
                                                    </select>
                                                    <small class="text-muted">Month when leave balance resets</small>
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Default Casual Leaves</label>
                                                    <input type="number" name="default_casual_leaves" class="form-control" 
                                                           value="<?php echo $settings['default_casual_leaves'] ?? 14; ?>" min="0" max="365">
                                                    <small class="text-muted">Default casual leaves per year</small>
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Default Sick Leaves</label>
                                                    <input type="number" name="default_sick_leaves" class="form-control" 
                                                           value="<?php echo $settings['default_sick_leaves'] ?? 7; ?>" min="0" max="365">
                                                    <small class="text-muted">Default sick leaves per year</small>
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Default Annual Leaves</label>
                                                    <input type="number" name="default_annual_leaves" class="form-control" 
                                                           value="<?php echo $settings['default_annual_leaves'] ?? 21; ?>" min="0" max="365">
                                                    <small class="text-muted">Default annual leaves per year</small>
                                                </div>
                                                <div class="col-12">
                                                    <button type="submit" name="update_leave" class="btn btn-primary">
                                                        <i class="bi bi-save"></i> Save Changes
                                                    </button>
                                                </div>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Email Settings -->
                            <div class="tab-pane fade" id="email" role="tabpanel">
                                <div class="card settings-card">
                                    <div class="card-header">
                                        <h6 class="mb-0">Email Settings</h6>
                                    </div>
                                    <div class="card-body">
                                        <form method="POST">
                                            <div class="row">
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">SMTP Host</label>
                                                    <input type="text" name="smtp_host" class="form-control" 
                                                           value="<?php echo $settings['smtp_host'] ?? 'smtp.gmail.com'; ?>">
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">SMTP Port</label>
                                                    <input type="number" name="smtp_port" class="form-control" 
                                                           value="<?php echo $settings['smtp_port'] ?? 587; ?>">
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">SMTP Username</label>
                                                    <input type="text" name="smtp_username" class="form-control" 
                                                           value="<?php echo $settings['smtp_username'] ?? ''; ?>">
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">SMTP Password</label>
                                                    <input type="password" name="smtp_password" class="form-control" 
                                                           value="<?php echo $settings['smtp_password'] ?? ''; ?>">
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">From Email</label>
                                                    <input type="email" name="from_email" class="form-control" 
                                                           value="<?php echo $settings['from_email'] ?? 'noreply@company.com'; ?>">
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">From Name</label>
                                                    <input type="text" name="from_name" class="form-control" 
                                                           value="<?php echo $settings['from_name'] ?? 'HR System'; ?>">
                                                </div>
                                                <div class="col-12">
                                                    <div class="alert alert-info">
                                                        <i class="bi bi-info-circle"></i> 
                                                        For Gmail, you may need to enable "Less secure app access" 
                                                        or use App-specific passwords.
                                                    </div>
                                                </div>
                                                <div class="col-12">
                                                    <button type="submit" name="update_email" class="btn btn-primary">
                                                        <i class="bi bi-save"></i> Save Changes
                                                    </button>
                                                    <button type="button" class="btn btn-secondary" onclick="testEmail()">
                                                        <i class="bi bi-envelope-check"></i> Test Email
                                                    </button>
                                                </div>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- System Settings -->
                            <div class="tab-pane fade" id="system" role="tabpanel">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="card settings-card">
                                            <div class="card-header">
                                                <h6 class="mb-0">System Information</h6>
                                            </div>
                                            <div class="card-body">
                                                <table class="table table-borderless">
                                                    <tr>
                                                        <td width="40%"><strong>PHP Version</strong></td>
                                                        <td><?php echo phpversion(); ?></td>
                                                    </tr>
                                                    <tr>
                                                        <td><strong>Database</strong></td>
                                                        <td>MySQL</td>
                                                    </tr>
                                                    <tr>
                                                        <td><strong>Server Software</strong></td>
                                                        <td><?php echo $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'; ?></td>
                                                    </tr>
                                                    <tr>
                                                        <td><strong>Server Name</strong></td>
                                                        <td><?php echo $_SERVER['SERVER_NAME'] ?? 'Unknown'; ?></td>
                                                    </tr>
                                                    <tr>
                                                        <td><strong>System Time</strong></td>
                                                        <td><?php echo date('Y-m-d H:i:s'); ?></td>
                                                    </tr>
                                                </table>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card settings-card">
                                            <div class="card-header">
                                                <h6 class="mb-0">System Maintenance</h6>
                                            </div>
                                            <div class="card-body">
                                                <div class="d-grid gap-2">
                                                    <button type="button" class="btn btn-outline-primary" onclick="clearCache()">
                                                        <i class="bi bi-trash"></i> Clear Cache
                                                    </button>
                                                    <button type="button" class="btn btn-outline-warning" onclick="backupDatabase()">
                                                        <i class="bi bi-database"></i> Backup Database
                                                    </button>
                                                    <button type="button" class="btn btn-outline-info" onclick="checkUpdates()">
                                                        <i class="bi bi-cloud-arrow-down"></i> Check for Updates
                                                    </button>
                                                    <button type="button" class="btn btn-outline-danger" onclick="resetSystem()">
                                                        <i class="bi bi-arrow-counterclockwise"></i> Reset System Settings
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="card settings-card mt-3">
                                    <div class="card-header">
                                        <h6 class="mb-0">Database Statistics</h6>
                                    </div>
                                    <div class="card-body">
                                        <div class="row">
                                            <?php
                                            // Get database statistics
                                            $tables = ['users', 'employees', 'attendance', 'leaves', 'departments'];
                                            foreach ($tables as $table):
                                                $stmt = $pdo->query("SELECT COUNT(*) as count FROM $table");
                                                $count = $stmt->fetch()['count'];
                                            ?>
                                            <div class="col-md-2 mb-3">
                                                <div class="card bg-light text-center">
                                                    <div class="card-body">
                                                        <h5 class="mb-0"><?php echo $count; ?></h5>
                                                        <small class="text-muted"><?php echo ucfirst($table); ?></small>
                                                    </div>
                                                </div>
                                            </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    function testEmail() {
        alert('Email test functionality would be implemented here.');
        // You would typically make an AJAX call to test email configuration
    }
    
    function clearCache() {
        if (confirm('Are you sure you want to clear all cache?')) {
            // AJAX call to clear cache
            alert('Cache cleared successfully!');
        }
    }
    
    function backupDatabase() {
        if (confirm('Create database backup?')) {
            window.location.href = 'backup_database.php';
        }
    }
    
    function checkUpdates() {
        alert('Checking for updates...');
        // AJAX call to check for updates
    }
    
    function resetSystem() {
        if (confirm('WARNING: This will reset all system settings to default values. Continue?')) {
            if (confirm('Are you absolutely sure? This action cannot be undone!')) {
                // AJAX call to reset system
                alert('System reset initiated. Page will reload.');
                setTimeout(() => location.reload(), 2000);
            }
        }
    }
    </script>
</body>
</html>